﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CfmUtility
{
    public class Program
    {
        public static void Main(string[] args)
        {
            if (args.Any() == false)
            {
                Console.WriteLine("Encode/Decode Nival Etherlords Camera format.");
                Console.WriteLine();
                Console.WriteLine("{0} filename", Process.GetCurrentProcess().ProcessName);
                Console.WriteLine();
                Console.WriteLine(" {0} file.cfm                decode file file.cfm into file.csv", Process.GetCurrentProcess().ProcessName);
                Console.WriteLine();
                Console.WriteLine(" {0} file.csv                encode file.csv into file.cfm", Process.GetCurrentProcess().ProcessName);
                Console.WriteLine();
                return;
            }

            var inputFile = args[0];
            var outputFileName = Path.GetFileNameWithoutExtension(inputFile);
            var extension = Path.GetExtension(inputFile);

            switch (extension.ToLower())
            {
                case ".csv":
                    Encode(inputFile, outputFileName + ".cmf");
                    break;
                
                case ".cmf":
                    Decode(inputFile, outputFileName + ".csv");
                    break;
            }
        }

        private static void Decode(string inputFile, string outputFile)
        {
            var buffer = File.ReadAllBytes(inputFile);
            using (var ms = new MemoryStream(buffer))
            {
                var reader = new BinaryReader(ms);
                var cfm = new Cfm();
                cfm.Read(reader);
                using (TextWriter writer = File.CreateText(outputFile))
                {
                    cfm.Write(writer);
                }
            }
        }

        private static void Encode(string inputFile, string outputFile)
        {
            using (var reader = File.OpenText(inputFile))
            {
                var cfm = new Cfm();
                cfm.Read(reader);
                using (var writer = new BinaryWriter(File.Open(outputFile, FileMode.Create)))
                {
                    cfm.Write(writer);
                }
            }
        }
    }

    internal class Cfm
    {
        private const uint Signature = 0x1567FBCA;

        private readonly List<Entry> entries = new List<Entry>();

        public void Write(BinaryWriter writer)
        {
            writer.Write(Signature);
            writer.Write(entries.Count - 2);
            foreach (var entry in this.entries)
            {
                entry.Write(writer);
            }
        }

        public void Write(TextWriter writer)
        {
            foreach (var entry in this.entries)
            {
                entry.Write(writer);
            }
        }

        public void Read(BinaryReader reader)
        {
            var sign = reader.ReadUInt32();

            if (Signature != sign)
            {
                var message = string.Format("Signature: {0}!={1}", Signature, sign);
                throw new InvalidOperationException(message);
            }

            this.entries.Clear();
            var count = reader.ReadInt32();
            this.entries.Add(Entry.Create(reader, 3));
            this.entries.Add(Entry.Create(reader, 3));
            for (var i = 0; i < count; i++)
            {
                this.entries.Add(Entry.Create(reader, 8));
            }
        }

        public void Read(TextReader reader)
        {
            this.entries.Clear();
            for (;;)
            {
                var line = reader.ReadLine();
                if (line == null)
                {
                    break;
                }

                this.entries.Add(Entry.Create(line));
            }
        }

        internal class Entry
        {
            private float[] Values { get; set; }

            public static Entry Create(BinaryReader reader, int count)
            {
                return new Entry()
                {
                    Values = Enumerable.Range(1, count).Select(x => reader.ReadSingle()).ToArray(),
                };
            }

            public static Entry Create(string line)
            {
                return new Entry()
                {
                    Values = line.Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries).Select(float.Parse).ToArray(),
                };
            }

            public void Write(TextWriter writer)
            {
                var line = string.Join(",", this.Values.Select(v => v.ToString()));
                writer.WriteLine(line);
            }

            public void Write(BinaryWriter writer)
            {
                foreach (var value in this.Values)
                {
                    writer.Write(value);
                }
            }
        }
    }
}
